<?php
namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class UserWebController extends Controller
{
    public function index()
    {
        // Statistics for Employees
        $totalEmployees = User::where('is_employee', true)->count();
        $activeEmployees = User::where('is_employee', true)->where('active', true)->count();

        $projectCounts = User::where('is_employee', true)
            ->whereNotNull('project_id')
            ->select('project_id', \Illuminate\Support\Facades\DB::raw('count(*) as total'))
            ->groupBy('project_id')
            ->pluck('total', 'project_id');

        $projectsWithStats = \App\Models\Project::orderBy('name')->get()->map(function ($p) use ($projectCounts) {
            $p->employees_count = $projectCounts[$p->id] ?? 0;
            return $p;
        });

        // List all users, excluding volunteers (they are managed in the Volunteers page)
        $query = User::where('is_volunteer', false)->with('roles')->orderBy('name');

        if (request('active') === '1') {
            $query->where('active', true);
        } elseif (request('active') === '0') {
            $query->where('active', false);
        }

        if (request('project_id')) {
            $query->where('project_id', request('project_id'));
        }

        if (request('type') === 'employee') {
            $query->where('is_employee', true);
        }

        $users = $query->paginate(20);

        // Add pending check for each user
        $users->each(function($u) {
            $u->pendingRequest = \App\Models\ChangeRequest::where('model_type', \App\Models\User::class)
                ->where('model_id', $u->id)
                ->where('status', 'pending')
                ->first();
        });

        return view('users.index', compact('users', 'totalEmployees', 'activeEmployees', 'projectsWithStats'));
    }
    public function create()
    {
        $roles = Role::orderBy('name')->get();
        return view('users.create', compact('roles'));
    }
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',
            'phone' => 'nullable|string',
            'is_employee' => 'boolean',
            'is_volunteer' => 'boolean',
            'active' => 'boolean',
            'roles' => 'array',
            'department' => 'nullable|string',
            'job_title' => 'nullable|string',
            'salary' => 'nullable|numeric',
            'join_date' => 'nullable|date',
            'contract_start_date' => 'nullable|date',
            'contract_end_date' => 'nullable|date|after:contract_start_date',
            'profile_photo' => 'nullable|image|max:10240',
            'contract_image' => 'nullable|image|max:10240',
            'criminal_record_image' => 'nullable|image|max:10240',
            'id_card_image' => 'nullable|image|max:10240',
            'annual_leave_quota' => 'nullable|integer',
            'leave_balance' => 'nullable|integer'
        ]);
        $data['password'] = Hash::make($data['password']);

        if ($request->hasFile('profile_photo')) {
            $data['profile_photo_path'] = $request->file('profile_photo')->store('profile-photos', 'public');
        }
        
        // Only set paths if files were uploaded to avoid nulling or object-to-string errors
        foreach(['contract_image', 'criminal_record_image', 'id_card_image'] as $field) {
            if ($request->hasFile($field)) {
                $data[$field] = $request->file($field)->store('user-docs', 'public');
            } else {
                unset($data[$field]);
            }
        }

        unset($data['profile_photo']);

        $user = User::create($data);
        if (!empty($data['roles'])) {
            $user->roles()->sync($data['roles']);
        }
        return redirect()->route('users.show', $user);
    }
    public function show(User $user)
    {
        $pendingRequest = \App\Models\ChangeRequest::where('model_type', User::class)
            ->where('model_id', $user->id)
            ->where('status', 'pending')
            ->first();

        return view('users.show', compact('user', 'pendingRequest'));
    }
    public function edit(User $user)
    {
        $pending = \App\Models\ChangeRequest::where('model_type', \App\Models\User::class)
            ->where('model_id', $user->id)
            ->where('status', 'pending')
            ->first();

        if ($pending) {
            return redirect()->route('change-requests.index')->with('info', 'هذا المستخدم لديه طلب مراجعة حالياً');
        }

        $roles = Role::orderBy('name')->get();
        return view('users.edit', compact('user', 'roles'));
    }
    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name' => 'sometimes|string',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:6',
            'phone' => 'nullable|string',
            'is_employee' => 'boolean',
            'is_volunteer' => 'boolean',
            'active' => 'boolean',
            'roles' => 'array',
            'department' => 'nullable|string',
            'job_title' => 'nullable|string',
            'salary' => 'nullable|numeric',
            'join_date' => 'nullable|date',
            'contract_start_date' => 'nullable|date',
            'contract_end_date' => 'nullable|date|after:contract_start_date',
            'profile_photo' => 'nullable|image|max:10240',
            'contract_image' => 'nullable|image|max:10240',
            'criminal_record_image' => 'nullable|image|max:10240',
            'id_card_image' => 'nullable|image|max:10240',
            'annual_leave_quota' => 'nullable|integer',
            'leave_balance' => 'nullable|integer'
        ]);
        if (!empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $user->update(['profile_photo_path' => $path]);
            unset($data['profile_photo_path']); // Don't include in change request
        }

        foreach(['contract_image', 'criminal_record_image', 'id_card_image'] as $field) {
            if ($request->hasFile($field)) {
                if ($user->$field) {
                    Storage::disk('public')->delete($user->$field);
                }
                $docPath = $request->file($field)->store('user-docs', 'public');
                $user->update([$field => $docPath]);
                unset($data[$field]); // Move to immediate Update
            } else {
                unset($data[$field]);
            }
        }

        unset($data['profile_photo']);

        $executor = function () use ($user, $data) {
            $user->update($data);
            if (isset($data['roles'])) {
                $user->roles()->sync($data['roles']);
            }
            return $user;
        };

        // If there is no data left to change besides images, just redirect
        if (empty($data) || (count($data) === 1 && isset($data['active']))) {
             // If ONLY images were changed (or active flag which we might want immediate too)
             return redirect()->route('users.show', $user)->with('success', 'تم تحديث الصور والبيانات بنجاح');
        }

        $result = \App\Services\ChangeRequestService::handleRequest(
            \App\Models\User::class,
            $user->id,
            'update',
            $data,
            $executor,
            true // Force Request for sensitive data
        );

        if ($result instanceof \App\Models\ChangeRequest && $result->status === 'pending') {
            return redirect()->route('users.show', $user)->with('success', 'تم تحديث الصور بنجاح، وباقي التعديلات بانتظار موافقة الإدارة');
        }

        return redirect()->route('users.show', $user)->with('success', 'تم تحديث بيانات المستخدم بنجاح');
    }
    public function destroy(User $user)
    {
        $executor = function () use ($user) {
            $user->delete();
            return true;
        };

        $result = \App\Services\ChangeRequestService::handleRequest(
            \App\Models\User::class,
            $user->id,
            'delete',
            [
                'note' => 'حذف مستخدم',
                'name' => $user->name,
                'job_title' => $user->job_title,
                'phone' => $user->phone,
                'is_employee' => $user->is_employee,
                'is_volunteer' => $user->is_volunteer,
            ],
            $executor,
            true // Force Request
        );

        if ($result instanceof \App\Models\ChangeRequest) {
            return redirect()->route('change-requests.index')->with('success', 'تم إرسال طلب حذف المستخدم للموافقة');
        }

        $user->delete();
        return redirect()->route('users.index')->with('success', 'تم حذف المستخدم بنجاح');
    }
    public function attachRole(User $user, Role $role)
    {
        $user->roles()->syncWithoutDetaching([$role->id]);
        return back();
    }
    public function detachRole(User $user, Role $role)
    {
        $user->roles()->detach($role->id);
        return back();
    }
}
